"use strict";
const baseDomain = "https://informea.org";
const baseURL = "https://buildingcircularity.org/wp-json/informea/v1/data";
const InitialKeywords = ['circular', 'economy']
const urlParams = new URL(document.location.toString()).searchParams;

const filter = {};
for (const [key, value] of urlParams.entries()) {
  const match = key.match(/^filter\[(.+)\]$/);
  if (match) {
    filter[match[1]] = value;
  }
}

let limit = 10;
if (urlParams.get("informea_page[limit]")) {
  limit = urlParams.get("informea_page[limit]");
}

let offset = 0;
if (urlParams.get("informea_page[offset]")) {
  offset = urlParams.get("informea_page[offset]");
}
let sort = "-search_api_relevance,field_weight,title";
if (urlParams.get("sort")) {
  sort = urlParams.get("sort");
}
let search = createFulltextQuery(InitialKeywords);
if (urlParams.get('searchtext')) {
  const urlText = decodeURIComponent(urlParams.get('searchtext'));
  const keywordArray = urlText.split(" ");
  search = createFulltextQuery([...InitialKeywords, ...keywordArray])
}
let params =
  `filter[type]=legislation&` +
  `filter[search_api_language]=en&` +
  `include=field_country,field_region,field_country.field_image,field_files,field_type_of_text,field_topics&`;

for (const key in filter) {
  params += `filter[${key}]=${filter[key]}&`;
}
params += search;
params += `page[limit]=${limit}&page[offset]=${offset}`;
params += `&sort=${sort}`;
params = encodeURIComponent(params);

const fetchParams = `?query=${params}`;
const fetchURL = baseURL + fetchParams;
let dateFacetValues = [];
fetch(fetchURL)
  .then((req) => {
    return req.json();
  })
  .then((data) => {
    hideLoading();
    initializeSearch();

    if (window.location.href.includes("?")) {
      showResetFilters()
    }
    const result = data.message;
    if (!result || result?.data?.length == 0) {
      noResults();
      return;
    }
    const container = document.querySelector(
      ".legislation-list .legislations"
    );
    const topInfo = document.querySelector(".legislation-list .top-info");
    const foundResults = createResultsFound(result.meta.count);
    topInfo.appendChild(foundResults);
    const sortLinks = createSortLinks();
    topInfo.appendChild(sortLinks);

    result.data.forEach((legislation) => {
      const id = "legislation-" + legislation.id;
      const teaser = createLegislationTeaser({
        id: id,
        title: legislation.attributes.title,
        titleUrl: baseDomain + legislation.attributes.path.alias,
        dateTime: legislation.attributes?.field_date
          ? new Date(legislation.attributes.field_date).toISOString()
          : null,
        displayDate: legislation.attributes?.field_date
          ? new Date(
              legislation.attributes.field_date
            ).toLocaleDateString("en-GB", {
              day: "2-digit",
              month: "short",
              year: "numeric",
            })
          : null,
        faolexUrl: legislation.attributes.field_faolex_url?.uri,
        ecolexUrl: legislation.attributes.field_ecolex_url?.uri,
      });
      container.appendChild(teaser);

      const country = result.included.filter((included) => {
        return (
          included.type == "node--country" &&
          included.id ==
            legislation.relationships.field_country.data[0]?.id
        );
      });
      updateCountryText(id, country, result.included);

      const region = result.included.filter((included) => {
        return (
          included.type == "taxonomy_term--regions" &&
          included.id ==
            legislation.relationships.field_region.data[0]?.id
        );
      });
      updateRegion(id, region);

      const downloadLinks = result.included.filter((included) => {
        return (
          included.type == "file--file" &&
          legislation.relationships.field_files.data
            .map((file) => file.id)
            .includes(included.id)
        );
      });
      updateBottomLinks(id, downloadLinks);

      const tag = result.included.filter((included) => {
        return (
          included.type == "taxonomy_term--types_of_text" &&
          included.id ==
            legislation.relationships.field_type_of_text.data[0]?.id
        );
      });
      updateTags(id, tag);

      const topics = result.included.filter((included) => {
        return (
          included.type == "taxonomy_term--topics" &&
          legislation.relationships.field_topics.data
            .map((file) => file.id)
            .includes(included.id)
        );
      });
      updateTopics(id, topics);
    });
    const facetsContainer = document.querySelector(".facets-container");
    const facetsOrder = [
      "jsonapi_region",
      "json_field_treaty",
      "jsonapi_search_party",
      "jsonapi_aggregated_date_int",
      "jsonapi_search_type_of_text",
    ];
    let facets = result.meta.facets.sort(function (a, b) {
      return facetsOrder.indexOf(a.id) - facetsOrder.indexOf(b.id);
    });
    facets = facets.filter((facet) => typeof facet == "object");
    facets.forEach((facet) => {
      if (facet.terms.length == 0) return;
      switch (facet.id) {
        case "jsonapi_region": {
          const facetContainer = createLinksFacetBlock(
            facet,
            "All regions",
            "Region"
          );
          facetsContainer.appendChild(facetContainer);
          return;
        }
        case "jsonapi_search_type_of_text": {
          const facetContainer = createLinksFacetBlock(
            facet,
            "All types of text",
            "Types of text"
          );
          facetsContainer.appendChild(facetContainer);
          return;
        }
        case "jsonapi_search_party": {
          const facetContainer = createSelectFacetBlock(
            facet,
            "In All parties",
            "Parties"
          );
          facetsContainer.appendChild(facetContainer);
          return;
        }
        case "json_field_treaty": {
          const facetContainer = createSelectFacetBlock(
            facet,
            "In all related treaties",
            "Treaties"
          );
          facetsContainer.appendChild(facetContainer);
          return;
        }
        case "jsonapi_aggregated_date_int": {
          const facetContainer = createDateSlider(facet, "Year / Period");
          facetsContainer.appendChild(facetContainer);
          return;
        }
      }
    });
    const domPagination = document.querySelector(".pagination");
    const pagination = initializePagination(result.meta.count);
    domPagination.appendChild(pagination);
  })
  .catch((e) => {
    hideLoading();
    initializeSearch();
    const facets = document.querySelector(".facets-container");
    facets.classList.add("hidden");
    const errorContainer = document.querySelector(".error");
    errorContainer.innerHTML =
      "<p>There was an error. Check developer console for more info</p>";
    console.error(e);
  });

function createLegislationTeaser({
  id,
  title,
  titleUrl,
  dateTime,
  displayDate,
  faolexUrl,
  ecolexUrl,
}) {
  const container = document.createElement("div");
  container.className = "legislation-teaser";
  container.id = id;
  const titleDiv = document.createElement("div");
  titleDiv.className = "legislation-title";
  titleDiv.innerHTML = `
<h2>
<a href="${titleUrl}">${title}</a>
</h2>
`;
  container.appendChild(titleDiv);

  const infoDiv = document.createElement("div");
  infoDiv.className = "legislation-info";
  infoDiv.innerHTML = `
<div class="legislation-flag">
<img class='hidden' src="" width="75" height="48" alt="" loading="lazy" class="img-fluid">
</div>
<div class="legislation-country">
<span class='hidden text'></span>
</div>
${
displayDate
  ? ' <div class="vertical-separator">|</div> <div class="legislation-date"><time datetime="' +
    dateTime +
    '">' +
    displayDate +
    "</time></div>"
  : ""
}

`;
  container.appendChild(infoDiv);

  const tagsDiv = document.createElement("div");
  tagsDiv.className = "legislation-tags";
  tagsDiv.innerHTML = `
<div class="topics">
<div class="legislation-region hidden"></div>
</div>
<div class="tags">

<span class="field_type_of_text hidden"></span>
${
  faolexUrl
    ? '<a class="source-link" href=' +
      faolexUrl +
      '><svg width="15" height="15" viewBox="0 0 15 15" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M8.66667 2.33997V3.50663H10.7608L5.02667 9.2408L5.84917 10.0633L11.5833 4.32913V6.4233H12.75V2.33997M11.5833 11.6733H3.41667V3.50663H7.5V2.33997H3.41667C2.76917 2.33997 2.25 2.86497 2.25 3.50663V11.6733C2.25 11.9827 2.37292 12.2795 2.59171 12.4983C2.8105 12.717 3.10725 12.84 3.41667 12.84H11.5833C11.8928 12.84 12.1895 12.717 12.4083 12.4983C12.6271 12.2795 12.75 11.9827 12.75 11.6733V7.58997H11.5833V11.6733Z" fill="#999999" style="fill:#999999;fill:color(display-p3 0.6000 0.6000 0.6000);fill-opacity:1;"/></svg>Faolex </a>'
    : ""
}
${
  ecolexUrl
    ? '<a class="source-link" href=' +
      ecolexUrl +
      '><svg width="15" height="15" viewBox="0 0 15 15" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M8.66667 2.33997V3.50663H10.7608L5.02667 9.2408L5.84917 10.0633L11.5833 4.32913V6.4233H12.75V2.33997M11.5833 11.6733H3.41667V3.50663H7.5V2.33997H3.41667C2.76917 2.33997 2.25 2.86497 2.25 3.50663V11.6733C2.25 11.9827 2.37292 12.2795 2.59171 12.4983C2.8105 12.717 3.10725 12.84 3.41667 12.84H11.5833C11.8928 12.84 12.1895 12.717 12.4083 12.4983C12.6271 12.2795 12.75 11.9827 12.75 11.6733V7.58997H11.5833V11.6733Z" fill="#999999" style="fill:#999999;fill:color(display-p3 0.6000 0.6000 0.6000);fill-opacity:1;"/></svg>Ecolex </a>'
    : ""
}
<div class="legislation-topics hidden"></div>

</div>
`;
  container.appendChild(tagsDiv);

  const bottomLinksDiv = document.createElement("div");
  bottomLinksDiv.className = "legislation-bottom-links";
  bottomLinksDiv.innerHTML = `
<span></span>
`;
  container.appendChild(bottomLinksDiv);

  return container;
}
function modifyFieldText(selector, text) {
  const DOMElement = document.querySelector(selector);
  DOMElement.innerHTML = text;
  DOMElement.classList.remove("hidden");
}
function updateCountryText(id, country, included) {
  if (!country || country.length == 0) return;
  const countryTextSelector = "#" + id + " .legislation-country .text";
  const text = country[0].attributes.title;
  modifyFieldText(countryTextSelector, text);

  const flag = included.filter((includedField) => {
    return (
      includedField.type == "file--file" &&
      includedField.id == country[0].relationships.field_image.data.id
    );
  });
  updateFlag(country[0], id, flag);
}

function updateFlag(country, id, flagField) {
  const flagSelector = "#" + id + " .legislation-flag img";
  const flag = document.querySelector(flagSelector);
  flag.classList.remove("hidden");
  flag.src = baseDomain + flagField[0].attributes.uri.url;
  flag.alt = country.attributes.title + " flag";
}
function updateRegion(id, region) {
  if (!region || region.length == 0) return;
  const regionTextSelector = "#" + id + " .topics .legislation-region";
  const text = region[0].attributes.name;
  modifyFieldText(regionTextSelector, text);
}
function updateBottomLinks(id, downloadLinks) {
  if (downloadLinks.length == 0) {
    hideBottomLinks(id);
    return;
  }
  const bottomLinks = downloadLinks.map((file) => {
    const url = file.attributes.uri.url;
    const link =
      '<a href="' +
      baseDomain +
      url +
      '" type="application/pdf" target="_blank">' +
      file.attributes.filename +
      "</a>";
    return link;
  });
  const bottomLinksContainerSelector =
    "#" + id + " .legislation-bottom-links";
  const bottomLinksContainer = document.querySelector(
    bottomLinksContainerSelector
  );
  bottomLinksContainer.innerHTML = `
${bottomLinks.join("")}
`;
}
function updateTags(id, tag) {
  if (!tag || tag.length == 0) return;
  const tagSelector = "#" + id + " .field_type_of_text";
  const iconAndName = `<svg width="15" height="15" viewBox="0 0 15 15" fill="none" xmlns="http://www.w3.org/2000/svg">
<g clip-path="url(#clip0_99_188)">
<path d="M12.75 3.75V4.91667H2.25V3.75H12.75ZM2.25 10.75H7.5V9.58333H2.25V10.75ZM2.25 7.83333H12.75V6.66667H2.25V7.83333Z" fill="#999999" style="fill:#999999;fill:color(display-p3 0.6000 0.6000 0.6000);fill-opacity:1;"/>
</g>
<defs>
<clipPath id="clip0_99_188">
<rect width="14" height="14" fill="white" style="fill:white;fill-opacity:1;" transform="translate(0.5 0.25)"/>
</clipPath>
</defs>
</svg><span>${tag[0].attributes.name}</span>`;
  modifyFieldText(tagSelector, iconAndName);
}
function updateTopics(id, topicsData) {
  if (!topicsData || topicsData.length == 0) return;
  const topicsSelector = "#" + id + " .legislation-topics";

  const topics = topicsData.map((topic) => {
    return `<div class="topic">${topic.attributes.name}</div>`;
  });
  const topicsContainer = document.querySelector(topicsSelector);

  topicsContainer.innerHTML = topics.join("");
  topicsContainer.classList.remove("hidden");
}
function createLinksFacetBlock(facet, text, title) {
  const container = document.createElement("div");
  container.className = "facet links";
  container.id = "facet-" + facet.id;
  const links = facet.terms.map((term) => {
    const link = `<a href='${getFacetLink(facet, term)}' class='${
      term.values.active ? "active" : ""
    }'>
  <span class='facet-label'>${term.values.label}</span>
  <span class='facet-count'> (${term.values.count})</span>
</a>`;
    return link;
  });
  const isAnyActive = facet.terms.some((term) => {
    return term.values.active;
  });
  if (text) {
    links.unshift(`<a href='${getLinkWithoutFacet(facet)}' class='${
      !isAnyActive ? "active" : ""
    }'>
  <span class='facet-label '>${text}</span>
</a>`);
  }
  if (title) {
    links.unshift(`<h3>${title}</h3>`);
  }
  container.innerHTML = links.join("");
  return container;
}
function getFacetLink(facet, term) {
  if (term.values.active) {
    return getLinkWithoutFacet(facet, term);
  }

  return (
    window.location.href.toString().split("?")[0] +
    "?" +
    sanitizeUrl(term.url.split("?")[1])
  );
}
function sanitizeUrl(url) {
  const decodeURL = decodeURIComponent(url).split('&');
  const URLWithoutPage = decodeURL.filter(filter => !filter.startsWith('page'))
  const URLWithoutFullTextSearch = URLWithoutPage.filter(filter => {
    return !filter.startsWith('filter[text-search') &&
    !filter.startsWith('filter[rendered-item') &&
    !filter.startsWith('filter[body-field') &&
    !filter.startsWith('filter[title-field') &&
    !filter.startsWith('filter[saa-field') 
  })
  const URLWithoutLanguage = URLWithoutFullTextSearch.filter(filter => !filter.startsWith('filter[search_api_language]'))
  const URLWithoutInclude = URLWithoutLanguage.filter(filter => !filter.startsWith('include'))
  const URLWithoutContentType = URLWithoutInclude.filter(filter => !filter.startsWith('filter[type]'))
  const encodedURL = encodeURI(URLWithoutContentType.join('&')).replace(/,/g, '%2C' );

  return encodedURL;
}
function getLinkWithoutFacet(facet, term) {
  let searchParams = new URLSearchParams(window.location.search);
  searchParams.delete(`filter[${facet.path}]`);
  const keysToDelete = [];
  for (let [key] of searchParams.entries()) {
    if (key.startsWith(`filter[${facet.path}-facet][condition][value]`)) {
      if (term) {
        if (searchParams.get(key) == term.values.value)
          searchParams.delete(key);
      } else {
        keysToDelete.push(key);
      }
    }
  }
  keysToDelete.forEach((key) => {
    searchParams.delete(key);
  });
  return (
    window.location.href.toString().split("?")[0] +
    "?" +
    searchParams.toString()
  );
}

function createSelectFacetBlock(facet, text, title) {
  const divContainer = document.createElement("div");
  const selectContainer = document.createElement("select");
  const container = document.createElement("optgroup");
  divContainer.className = "facet selects";
  divContainer.id = "facet-" + facet.id;
  const terms = facet.terms.sort((a, b) => (a.values.active ? -1 : 1));
  let selectOptions = terms.map((term) => {
    const option = `<option value='${getFacetLink(facet, term)}'>${
      term.values.label
    } (${term.values.count})</option>`;
    return option;
  });
  if (facet.terms.some((term) => term.values.active)) {
    selectOptions.splice(
      1,
      0,
      `<option value='${getLinkWithoutFacet(facet)}'>${text}</option>`
    );
  } else {
    selectOptions.unshift(
      `<option value='${getLinkWithoutFacet(facet)}'>${text}</option>`
    );
  }
  if (title) {
    divContainer.innerHTML = `<h3>${title}</h3>`;
  }
  container.innerHTML = selectOptions.join("");
  selectContainer.appendChild(container);
  divContainer.appendChild(selectContainer);

  selectContainer.addEventListener("change", (e) => {
    window.location.href = e.target.value;
  });
  return divContainer;
}
function createResultsFound(count) {
  const container = document.createElement("div");
  container.textContent = `Found ${count} results`;
  return container;
}
function initializePagination(count) {
  const searchParams = new URLSearchParams(window.location.search);
  const paramPageLimit = searchParams.get("informea_page[limit]") ?? 10;
  const paramPageOffset = searchParams.get("informea_page[offset]") ?? 0;
  const lastPage = getLastPage(count, paramPageLimit);
  const currentPage = Math.floor(paramPageOffset / paramPageLimit) + 1;
  const container = document.createElement("div");
  const pageElements = [];
  let loopFirstPage = currentPage > 4 ? currentPage - 4 : 1;
  let loopLastPage =
    lastPage - currentPage < 4 ? lastPage : currentPage + 4;
  if (lastPage - currentPage < 4 && lastPage > 4)
    loopFirstPage -= 4 - (lastPage - currentPage);
  if (currentPage < 5) loopLastPage += 5 - currentPage;
  if (loopLastPage > lastPage) loopLastPage = lastPage;
  if (loopFirstPage < 1) loopFirstPage = 1;
  for (let i = loopFirstPage; i <= loopLastPage; i++) {
    if (i == currentPage) {
      pageElements.push(
        `<a class='current' href='${getPaginationHref(i)}'>${i}</a>`
      );
    } else {
      pageElements.push(`<a href='${getPaginationHref(i)}'>${i}</a>`);
    }
  }
  if (currentPage > 5 && lastPage > 9) {
    pageElements.unshift("<a >...</a>");
  }
  if (lastPage - currentPage > 4 && lastPage > 9) {
    pageElements.push("<a>...</a>");
  }
  if (currentPage != 1) {
    pageElements.unshift(`<a class="pagination-aditional" href='${getPaginationHref(
      currentPage - 1
    )}'><svg width="6" height="9" viewBox="0 0 6 9" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M5.71997 1.63995L2.66664 4.69995L5.71997 7.75995L4.77997 8.69995L0.77997 4.69995L4.77997 0.699951L5.71997 1.63995Z" fill="#111111" style="fill:#111111;fill-opacity:1;"/>
</svg></a>`);
    pageElements.unshift(`<a class="pagination-aditional" href='${getPaginationHref(
      1
    )}'><svg width="6" height="9" viewBox="0 0 6 9" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M5.71997 1.63995L2.66664 4.69995L5.71997 7.75995L4.77997 8.69995L0.77997 4.69995L4.77997 0.699951L5.71997 1.63995Z" fill="#111111" style="fill:#111111;fill-opacity:1;"/>
</svg><svg width="6" height="9" viewBox="0 0 6 9" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M5.71997 1.63995L2.66664 4.69995L5.71997 7.75995L4.77997 8.69995L0.77997 4.69995L4.77997 0.699951L5.71997 1.63995Z" fill="#111111" style="fill:#111111;fill-opacity:1;"/>
</svg></a>`);
  }
  if (lastPage != currentPage) {
    pageElements.push(`<a class="pagination-aditional" href='${getPaginationHref(
      currentPage + 1
    )}'><svg width="6" height="9" viewBox="0 0 6 9" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M0.780029 7.75995L3.83336 4.69995L0.780029 1.63995L1.72003 0.699951L5.72003 4.69995L1.72003 8.69995L0.780029 7.75995Z" fill="#111111" style="fill:#111111;fill:color(display-p3 0.0667 0.0667 0.0667);fill-opacity:1;"/>
</svg></a>`);
    pageElements.push(`<a class="pagination-aditional" href='${getPaginationHref(
      lastPage
    )}'><svg width="6" height="9" viewBox="0 0 6 9" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M0.780029 7.75995L3.83336 4.69995L0.780029 1.63995L1.72003 0.699951L5.72003 4.69995L1.72003 8.69995L0.780029 7.75995Z" fill="#111111" style="fill:#111111;fill:color(display-p3 0.0667 0.0667 0.0667);fill-opacity:1;"/>
</svg><svg width="6" height="9" viewBox="0 0 6 9" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M0.780029 7.75995L3.83336 4.69995L0.780029 1.63995L1.72003 0.699951L5.72003 4.69995L1.72003 8.69995L0.780029 7.75995Z" fill="#111111" style="fill:#111111;fill:color(display-p3 0.0667 0.0667 0.0667);fill-opacity:1;"/>
</svg></a>`);
  }
  container.innerHTML = pageElements.join("");
  return container;
}
function getLastPage(totalItems, limit) {
  return Math.ceil(totalItems / limit);
}
function getPaginationHref(i) {
  const searchParams = new URLSearchParams(window.location.search);
  const paramPageLimit = parseInt(
    searchParams.get("informea_page[limit]") ?? 10
  );
  searchParams.set("informea_page[offset]", (i - 1) * paramPageLimit);
  return (
    window.location.href.toString().split("?")[0] +
    "?" +
    searchParams.toString()
  );
}
function createSortLinks() {
  const searchParams = new URLSearchParams(window.location.search);
  const currentUrl = window.location.href.toString().split("?")[0];

  const select = document.createElement("select");

  const options = [
    { value: "-search_api_relevance,field_weight,title", text: "Relevance" },
    { value: "-aggregated_date,field_weight,title", text: "Most Recent" },
  ];

  options.forEach((option) => {
    const optElement = document.createElement("option");
    optElement.value = option.value;
    optElement.textContent = option.text;
    if (searchParams.get("sort") === option.value) {
      optElement.selected = true;
    }
    select.appendChild(optElement);
  });

  select.addEventListener("change", function () {
    searchParams.set("sort", this.value);
    window.location.href = currentUrl + "?" + searchParams.toString();
  });

  const container = document.createElement("div");
  container.classList.add("sort-container");
  container.innerHTML = "<span>Sort by</span>";
  container.appendChild(select);
  return container;
}
function createDateSlider(facet, title) {
  const container = document.createElement("div");
  container.className = "facet slider";
  container.id = "facet-" + facet.id;
  if (title) {
    container.innerHTML = `<h3>${title}</h3>`;
  }
  dateFacetValues = facet.terms.map((term) => {
    return term.values.value;
  });
  dateFacetValues.sort((a, b) => a - b);
  const beggingYear = parseInt(dateFacetValues[0]);
  const endYear = parseInt(dateFacetValues[dateFacetValues.length - 1]);
  const silderContainer = document.createElement("div");
  const searchParams = new URLSearchParams(window.location.search);
  const min =
    searchParams.get("filter[date-facet][condition][value][0]") ??
    beggingYear;
  const max =
    searchParams.get("filter[date-facet][condition][value][1]") ??
    endYear;

  noUiSlider.create(silderContainer, {
    start: [min, max],
    connect: true,
    range: {
      min: 0,
      max: facet.terms.length - 1,
    },
    step: 1,
    tooltips: true,
    format: {
      to: function (value) {
        return dateFacetValues[Math.round(value)];
      },
      from: function (value) {
        return dateFacetValues.indexOf(value);
      },
    },
  });
  silderContainer.noUiSlider.on("change", function (values, handle) {
    const startYear = values[0];
    const endYear = values[1];
    let searchParams = new URLSearchParams(window.location.search);
    searchParams.set(
      "filter[date-facet][condition][path]",
      `aggregated_date_int`
    );
    searchParams.set(
      "filter[date-facet][condition][operator]",
      `BETWEEN`
    );
    searchParams.set(
      "filter[date-facet][condition][value][0]",
      startYear
    );
    searchParams.set("filter[date-facet][condition][value][1]", endYear);

    window.location.href =
      window.location.href.toString().split("?")[0] +
      "?" +
      searchParams.toString();
  });
  container.appendChild(silderContainer);
  return container;
}
function hideBottomLinks(id) {
  const bottomLinksSelector = "#" + id + " .legislation-bottom-links";
  const bottomLinksElement = document.querySelector(bottomLinksSelector);
  bottomLinksElement.classList.add("hidden");
}
function noResults() {
  const facets = document.querySelector(".facets-container");
  facets.classList.add("hidden");
  const errorContainer = document.querySelector(".error");
  errorContainer.innerHTML = "<p>There are no results</p>";
}
function hideLoading() {
  const loadingContainer = document.querySelector('.loading-container');
  loadingContainer.classList.add('hidden');
  const disclaimer = document.querySelector(".disclaimer")
  disclaimer.classList.remove('hidden');
}
function showResetFilters () {
  const showFiltersButton = document.querySelector("#clear-filters");
  if (!showFiltersButton) return;
  showFiltersButton.classList.remove('hidden');
}
function createFulltextQuery(keywords) {
  let query = '';
  keywords.forEach((keyword, index) => {
  query += `filter[text-search${index}][group][conjunction]=OR&` +
  `filter[rendered-item${index}][condition][path]=rendered_item_teaser&`+
  `filter[rendered-item${index}][condition][operator]=IN&`+
  `filter[rendered-item${index}][condition][value]=${keyword}&`+
  `filter[rendered-item${index}][condition][memberOf]=text-search${index}&`+
  `filter[body-field${index}][condition][path]=body&`+
  `filter[body-field${index}][condition][operator]=IN&`+
  `filter[body-field${index}][condition][value]=${keyword}&`+
  `filter[body-field${index}][condition][memberOf]=text-search${index}&`+
  `filter[title-field${index}][condition][path]=title&`+
  `filter[title-field${index}][condition][operator]=IN&`+
  `filter[title-field${index}][condition][value]=${keyword}&`+
  `filter[title-field${index}][condition][memberOf]=text-search${index}&`+
  `filter[saa-field${index}][condition][path]=saa_field_files&`+
  `filter[saa-field${index}][condition][operator]=IN&`+
  `filter[saa-field${index}][condition][value]=${keyword}&`+
  `filter[saa-field${index}][condition][memberOf]=text-search${index}&`
  })
 return query;
}
const searchForm = document.querySelector(".search-container form");
searchForm.addEventListener("submit", (e) => {
  const value = e.target.querySelector("input").value;
  const searchParams = new URLSearchParams(window.location.search);
  const currentUrl = window.location.href.toString().split("?")[0];
  searchParams.set("searchtext", value);
  window.location.href = currentUrl + "?" + searchParams.toString();
})
function initializeSearch() {
  const searchContainer = document.querySelector(".search-container");
  searchContainer.classList.remove('hidden');
  if (!urlParams.get("searchtext")) {
    return;
  }
  const searchInput = document.querySelector(".search-container input");
  searchInput.value = urlParams.get("searchtext");
}